// VVV Extension Popup

const API_BASE = 'https://vvv.jkjk.uk';
const DEFAULT_GROUP_ID = '70000';

// DOM要素
const groupSelect = document.getElementById('group-select');
const saveFavBtn = document.getElementById('save-fav');
const saveTodoBtn = document.getElementById('save-todo');
const toastEl = document.getElementById('toast');
const versionEl = document.getElementById('version');

// バージョン表示 & 更新チェック
chrome.management.getSelf(async (info) => {
  const currentVersion = info.version;
  versionEl.textContent = `v${currentVersion}`;

  // サーバーから最新バージョンを取得
  try {
    const res = await fetch(`${API_BASE}/extension/version`);
    if (res.ok) {
      const data = await res.json();
      const latestVersion = data.version;

      // バージョン比較（セマンティックバージョニング）
      if (compareVersions(latestVersion, currentVersion) > 0) {
        showUpdateBanner(currentVersion, latestVersion);
      }
    }
  } catch (e) {
    // サーバー接続失敗は無視
  }
});

// 更新バナー表示
function showUpdateBanner(current, latest) {
  const banner = document.createElement('div');
  banner.id = 'update-banner';
  banner.innerHTML = `
    <div style="background: linear-gradient(135deg, #ffc107, #ff9800); color: #000; padding: 8px; border-radius: 6px; margin-bottom: 10px; text-align: center; font-size: 12px;">
      <div style="font-weight: bold;">🆕 v${current} → v${latest}</div>
      <button id="update-btn" style="margin-top: 6px; padding: 4px 12px; border: none; border-radius: 4px; background: #000; color: #fff; cursor: pointer; font-size: 11px;">
        ダウンロードして更新
      </button>
    </div>
  `;
  document.body.insertBefore(banner, document.body.firstChild);

  document.getElementById('update-btn').addEventListener('click', async () => {
    // ZIPをダウンロード
    const link = document.createElement('a');
    link.href = `${API_BASE}/extension/download`;
    link.click();

    // 更新手順を表示
    banner.innerHTML = `
      <div style="background: #17a2b8; color: #fff; padding: 10px; border-radius: 6px; margin-bottom: 10px; font-size: 11px; line-height: 1.5;">
        <div style="font-weight: bold; margin-bottom: 6px;">📦 更新手順:</div>
        <div>1. ZIPを解凍して既存フォルダを上書き</div>
        <div>2. <a href="#" id="open-extensions" style="color: #fff; text-decoration: underline;">chrome://extensions</a> を開く</div>
        <div>3. この拡張の🔄ボタンをクリック</div>
      </div>
    `;

    document.getElementById('open-extensions').addEventListener('click', (e) => {
      e.preventDefault();
      chrome.tabs.create({ url: 'chrome://extensions' });
    });
  });

  versionEl.style.display = 'none';
}

// バージョン比較関数 (1: a > b, 0: a == b, -1: a < b)
function compareVersions(a, b) {
  const pa = a.split('.').map(Number);
  const pb = b.split('.').map(Number);
  for (let i = 0; i < Math.max(pa.length, pb.length); i++) {
    const na = pa[i] || 0;
    const nb = pb[i] || 0;
    if (na > nb) return 1;
    if (na < nb) return -1;
  }
  return 0;
}

// AmazonリンクからアフィリエイトタグとトラッキングIDを除去
function cleanAmazonUrl(url) {
  try {
    const urlObj = new URL(url);
    // Amazon判定 (amazon.co.jp, amazon.com, etc.)
    if (!urlObj.hostname.includes('amazon.')) {
      return url;
    }

    // 除去するパラメータ
    const removeParams = [
      'tag', 'ref', 'ref_', 'linkCode', 'linkId', 'ascsubtag',
      'psc', 'smid', 'sourceid', 'sr', 'qid', 'keywords', 'sprefix',
      'crid', 'dib', 'dib_tag', 'th', 'language'
    ];
    // pf_rd_*, pd_rd_* も除去
    const prefixParams = ['pf_rd_', 'pd_rd_'];

    const params = urlObj.searchParams;
    const keysToDelete = [];

    for (const key of params.keys()) {
      if (removeParams.includes(key) || prefixParams.some(p => key.startsWith(p))) {
        keysToDelete.push(key);
      }
    }

    keysToDelete.forEach(key => params.delete(key));

    // /ref=xxx/ パス部分も除去
    urlObj.pathname = urlObj.pathname.replace(/\/ref=[^/]+/, '');

    return urlObj.toString();
  } catch (e) {
    return url;
  }
}

// トースト表示関数
function showToast(message, type = 'loading', icon = '') {
  toastEl.className = `toast show ${type}`;
  toastEl.innerHTML = icon ? `<span class="toast-icon">${icon}</span>${message}` : message;
}

function hideToast() {
  toastEl.className = 'toast';
}

// 保存済みグループ選択を復元
chrome.storage.local.get('selectedGroupId', (result) => {
  if (result.selectedGroupId) {
    groupSelect.value = result.selectedGroupId;
  }
});

// グループ選択変更時に保存
groupSelect.addEventListener('change', () => {
  chrome.storage.local.set({ selectedGroupId: groupSelect.value });
});

// イベントリスナー登録
saveFavBtn.addEventListener('click', () => saveToVVV('favorites'));
saveTodoBtn.addEventListener('click', () => saveToVVV('todo'));

async function saveToVVV(saveType) {
  showToast('保存中...', 'loading', '⏳');

  try {
    // 現在のタブを取得
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    const url = cleanAmazonUrl(tab.url);  // Amazonリンクをクリーニング
    const title = tab.title || url;

    // トークンを取得
    const { accessToken } = await chrome.storage.local.get('accessToken');

    if (!accessToken) {
      // ログインページを開く
      chrome.tabs.create({ url: `${API_BASE}/login?extension=1` });
      window.close();
      return;
    }

    // 選択されたグループIDを取得
    const selectedGroupId = parseInt(groupSelect.value, 10) || parseInt(DEFAULT_GROUP_ID, 10);

    // エンドポイントを選択
    const endpoint = saveType === 'todo'
      ? `${API_BASE}/api/extension/save-todo`
      : `${API_BASE}/api/extension/save`;

    // APIリクエスト
    const response = await fetch(endpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${accessToken}`
      },
      body: JSON.stringify({
        href: url,
        title: title,
        src: url,
        group_id: selectedGroupId
      })
    });

    if (response.status === 401) {
      // トークン期限切れ
      chrome.storage.local.remove('accessToken');
      chrome.tabs.create({ url: `${API_BASE}/login?extension=1` });
      window.close();
      return;
    }

    if (response.ok) {
      const data = await response.json();
      const typeLabel = saveType === 'todo' ? 'TODO' : 'お気に入り';

      // 重複チェック
      const isDuplicate = data.message?.includes('Already') || data.message?.includes('既に');

      if (isDuplicate) {
        // 重複の場合は警告表示
        showToast(`既に${typeLabel}に登録済み`, 'warning', '⚠️');
        chrome.action.setBadgeText({ text: '!', tabId: tab.id });
        chrome.action.setBadgeBackgroundColor({ color: '#ff9800' });
      } else {
        // 新規保存成功
        showToast(`${typeLabel}に保存しました`, 'success', '✓');
        const badgeText = saveType === 'todo' ? '☑' : '★';
        const badgeColor = saveType === 'todo' ? '#17a2b8' : '#4CAF50';
        chrome.action.setBadgeText({ text: badgeText, tabId: tab.id });
        chrome.action.setBadgeBackgroundColor({ color: badgeColor });
      }

      // 1.5秒後にポップアップを閉じる
      setTimeout(() => {
        chrome.action.setBadgeText({ text: '', tabId: tab.id });
        window.close();
      }, 1500);
    } else {
      const error = await response.json();
      showToast(error.detail || 'エラーが発生しました', 'error', '✗');
    }
  } catch (err) {
    console.error('VVV Extension Error:', err);
    showToast('ネットワークエラー', 'error', '✗');
  }
}

// ============ インポート機能 ============

const importBookmarksBtn = document.getElementById('import-bookmarks');
const importChannelsBtn = document.getElementById('import-channels');

importBookmarksBtn.addEventListener('click', importChromeBookmarks);
importChannelsBtn.addEventListener('click', importYouTubeChannels);

// フォルダを開くボタン
document.getElementById('open-folder').addEventListener('click', async (e) => {
  e.preventDefault();
  const extensionId = chrome.runtime.id;

  // Windowsの場合のパスをクリップボードにコピー
  const winPath = `%LOCALAPPDATA%\\Google\\Chrome\\User Data\\Default\\Extensions\\${extensionId}`;

  try {
    await navigator.clipboard.writeText(extensionId);
    showToast(`ID: ${extensionId.slice(0, 8)}... コピー済`, 'success', '📋');

    // 拡張機能ページを開く
    setTimeout(() => {
      chrome.tabs.create({ url: `chrome://extensions/?id=${extensionId}` });
    }, 1000);
  } catch (err) {
    // クリップボード失敗時は直接開く
    chrome.tabs.create({ url: `chrome://extensions/?id=${extensionId}` });
  }
});

// Chromeブックマークを再帰的に取得
function getAllBookmarks(nodes, result = []) {
  for (const node of nodes) {
    if (node.url) {
      result.push({ url: cleanAmazonUrl(node.url), title: node.title || node.url });
    }
    if (node.children) {
      getAllBookmarks(node.children, result);
    }
  }
  return result;
}

async function importChromeBookmarks() {
  const { accessToken } = await chrome.storage.local.get('accessToken');
  if (!accessToken) {
    chrome.tabs.create({ url: `${API_BASE}/login?extension=1` });
    window.close();
    return;
  }

  importBookmarksBtn.disabled = true;
  showToast('ブックマーク取得中...', 'loading', '⏳');

  try {
    const tree = await chrome.bookmarks.getTree();
    const bookmarks = getAllBookmarks(tree);

    if (bookmarks.length === 0) {
      showToast('ブックマークがありません', 'warning', '⚠️');
      importBookmarksBtn.disabled = false;
      return;
    }

    showToast(`${bookmarks.length}件をインポート中...`, 'loading', '⏳');

    const selectedGroupId = parseInt(groupSelect.value, 10) || 70000;
    const response = await fetch(`${API_BASE}/api/extension/import-bookmarks`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${accessToken}`
      },
      body: JSON.stringify({
        bookmarks: bookmarks,
        group_id: selectedGroupId
      })
    });

    if (response.ok) {
      const data = await response.json();
      showToast(data.message, 'success', '✓');
    } else {
      const error = await response.json();
      showToast(error.detail || 'エラー', 'error', '✗');
    }
  } catch (err) {
    console.error('Import error:', err);
    showToast('インポートエラー', 'error', '✗');
  } finally {
    importBookmarksBtn.disabled = false;
  }
}

async function importYouTubeChannels() {
  const { accessToken } = await chrome.storage.local.get('accessToken');
  if (!accessToken) {
    chrome.tabs.create({ url: `${API_BASE}/login?extension=1` });
    window.close();
    return;
  }

  // 現在のタブがYouTubeのチャンネルページか確認
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab.url.includes('youtube.com/feed/channels')) {
    showToast('YouTubeの購読チャンネル一覧ページを開いてください', 'warning', '⚠️');
    chrome.tabs.create({ url: 'https://www.youtube.com/feed/channels' });
    return;
  }

  importChannelsBtn.disabled = true;
  showToast('チャンネル取得中...', 'loading', '⏳');

  try {
    // コンテンツスクリプトでチャンネル情報を取得
    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: () => {
        const channels = [];
        // チャンネルリンクを取得
        const links = document.querySelectorAll('a#main-link[href*="/channel/"], a#main-link[href*="/@"]');
        links.forEach(link => {
          const href = link.href;
          let channelId = '';
          if (href.includes('/channel/')) {
            channelId = href.split('/channel/')[1].split(/[/?]/)[0];
          } else if (href.includes('/@')) {
            channelId = '@' + href.split('/@')[1].split(/[/?]/)[0];
          }
          const titleEl = link.querySelector('#channel-title, #text');
          const title = titleEl?.textContent?.trim() || channelId;
          if (channelId && !channels.some(c => c.channel_id === channelId)) {
            channels.push({ channel_id: channelId, title });
          }
        });
        return channels;
      }
    });

    const channels = results[0]?.result || [];
    if (channels.length === 0) {
      showToast('チャンネルが見つかりません。ページをスクロールしてください', 'warning', '⚠️');
      importChannelsBtn.disabled = false;
      return;
    }

    showToast(`${channels.length}件をインポート中...`, 'loading', '⏳');

    const response = await fetch(`${API_BASE}/api/extension/import-channels`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${accessToken}`
      },
      body: JSON.stringify({ channels, user_id: 65510 })
    });

    if (response.ok) {
      const data = await response.json();
      showToast(data.message, 'success', '✓');
    } else {
      const error = await response.json();
      showToast(error.detail || 'エラー', 'error', '✗');
    }
  } catch (err) {
    console.error('Import error:', err);
    showToast('インポートエラー', 'error', '✗');
  } finally {
    importChannelsBtn.disabled = false;
  }
}
